/* Wrapper/Extension code to libusb-0.1 to support asynchronous requests
 * on Linux platforns 
 *
 * (C) 2004 by Harald Welte <laforge@gnumonks.org>
 *
 * Distributed under the terms of GNU LGPL, Version 2.1
 */

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/utsname.h>
#include <sys/ioctl.h>

#include "ausb.h"

#ifdef DEBUG_AUSB
#define DEBUGP(x, args...)	fprintf(stderr, "%s:%s():%u " x, __FILE__, \
					__FUNCTION__, __LINE__, ## args)
#else
#define DEBUGP(x, args...);
#endif

static int ausb_get_fd(ausb_dev_handle *ah)
{
	return *((int *)ah->uh);
}


static int kernel_2_5;

/* FIXME: this has to go */
static struct ausb_dev_handle *last_handle = NULL;

static int is_kernel_2_5()
{
	struct utsname uts;
	int ret;

	uname(&uts);

	ret = (strncmp(uts.release, "2.5.", 4) == 0) ||
	       (strncmp(uts.release, "2.6.", 4) == 0);

	return ret;
}

void ausb_dump_urb(struct usbdevfs_urb *uurb)
{
	DEBUGP("urb(%p): type=%u, endpoint=0x%x, status=%d, flags=0x%x, number_of_packets=%d, error_count=%d\n", uurb, uurb->type, uurb->endpoint, uurb->status, uurb->flags, uurb->number_of_packets, uurb->error_count);
}

void ausb_fill_int_urb(struct usbdevfs_urb *uurb, unsigned char endpoint,
		      void *buffer, int buffer_length)
				    
{
	uurb->type = kernel_2_5 ? USBDEVFS_URB_TYPE_INTERRUPT : USBDEVFS_URB_TYPE_BULK;
	uurb->endpoint = endpoint; /* | USB_DIR_IN; */
	uurb->flags = kernel_2_5 ? 0 : 1 ; /* USBDEVFS_URB_QUEUE_BULK; */
	uurb->buffer = buffer;
	uurb->buffer_length = buffer_length;
	uurb->signr = SIGRTMIN;
	uurb->start_frame = -1;
}

int ausb_submit_urb(ausb_dev_handle *ah, struct usbdevfs_urb *uurb)
{
	int ret;

	DEBUGP("ausb_submit_urb(ah=%p)\n", ah);

	/* save ausb_dev_handle in opaque usercontext field */
	uurb->usercontext = ah;

	do {
		ret = ioctl(ausb_get_fd(ah), USBDEVFS_SUBMITURB, uurb);
	} while (ret < 0 && errno == EINTR);

	return ret;
}

struct usbdevfs_urb *ausb_get_urb(ausb_dev_handle *ah)
{
	int ret;
	struct usbdevfs_urb *uurb;

	do {
		ret = ioctl(ausb_get_fd(ah), USBDEVFS_REAPURBNDELAY, &uurb);
	} while (ret < 0 && errno == EINTR);

	if (ret < 0 && errno == EAGAIN)
		return NULL;

	return uurb;
}

static void handle_urb(struct usbdevfs_urb *uurb)
{
	struct ausb_dev_handle *ah = uurb->usercontext;

	DEBUGP("called, ah=%p\n");

	if (uurb->type >= AUSB_USBDEVFS_URB_TYPES) {
		DEBUGP("unknown uurb type %u\n", uurb->type);
		return;
	}

	if (!ah->cb[uurb->type].handler) {
		DEBUGP("received URB type %u, but no handler\n", uurb->type);
		return;
	}
	ah->cb[uurb->type].handler(uurb, ah->cb[uurb->type].userdata);
}

static void sigact_rtmin(int sig, siginfo_t *siginfo, void *v)
{
	int count;
	struct usbdevfs_urb *urb;

	if (sig != SIGRTMIN)
		return;
 
	DEBUGP("errno=%d, si_addr=%p\n", siginfo->errno, siginfo->si_addr);

	if (!last_handle)
		return;

	for (count = 0; ; count++) {
		urb = ausb_get_urb(last_handle);

		if (urb == NULL) {
			DEBUGP("ausb_get_urb() returned urb==NULL\n");
			break;
		}

		handle_urb(urb);
	}
}


int ausb_init(void)
{
	struct sigaction act;

	memset(&act, 0, sizeof(act));
	act.sa_sigaction = sigact_rtmin;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_SIGINFO;

	sigaction(SIGRTMIN, &act, NULL);

	kernel_2_5 = is_kernel_2_5();

	usb_init();
	usb_find_busses();
	usb_find_devices();
	return 1;
}

ausb_dev_handle *ausb_open(struct usb_device *dev)
{
	ausb_dev_handle *dh = malloc(sizeof *dh);

	memset(dh, 0, sizeof(*dh));

	dh->uh = usb_open(dev);

	if (!dh->uh) {
		free(dh);
		return NULL;
	}

	last_handle = dh;

	return dh;
}

int ausb_register_callback(ausb_dev_handle *ah, unsigned char type,
			   void (*callback)(struct usbdevfs_urb *uurb,
					    void *userdata),
			   void *userdata)
{
	DEBUGP("registering callback for type %u:%p\n", type, callback);
	if (type >= AUSB_USBDEVFS_URB_TYPES) {
		errno = EINVAL;
		return -1;
	}

	if (!kernel_2_5 && type == USBDEVFS_URB_TYPE_INTERRUPT)
		type = USBDEVFS_URB_TYPE_BULK;

	ah->cb[type].handler = callback;
	ah->cb[type].userdata = userdata;

	return 0;
}

int ausb_claim_interface(ausb_dev_handle *ah, int interface)
{
	return usb_claim_interface(ah->uh, interface);
}

int ausb_release_interface(ausb_dev_handle *ah, int interface)
{
	return usb_release_interface(ah->uh, interface);
}

int ausb_set_configuration(ausb_dev_handle *ah, int configuration)
{
	return usb_set_configuration(ah->uh, configuration);
}

int ausb_bulk_write(ausb_dev_handle *ah, int ep, char *bytes, int size, int timeout)
{
	return usb_bulk_write(ah->uh, ep, bytes, size, timeout);
}

int ausb_bulk_read(ausb_dev_handle *ah, int ep, char *bytes, int size, int timeout)
{
	return usb_bulk_read(ah->uh, ep, bytes, size, timeout);
}

int ausb_close(struct ausb_dev_handle *ah)
{
	if (ah == last_handle)
		last_handle = NULL;

	return usb_close(ah->uh);
}


