/***************************************************************************
 * CT-API library for the REINER SCT cyberJack pinpad/e-com USB.
 * Copyright (C) 2004  REINER SCT
 * Author: Harald Welte
 * Support: linux-usb@sii.li
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * File: cjctapi_switch.c
 * CVS: $Id:$
 ***************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <usb.h>

#include "ecom/ctapi-ecom.h"
#include "ecom/ctapi.h"
#include "ppa/cjppa.h"

#include "ctn_list.h"

#define CJPPA_USB_VENDOR_ID	0x0c4b

#define CJECOM_USB_DEVICE_ID	0x0100
#define CJPPA_USB_DEVICE_ID	0x0300

#define cjctapi_enter() cjctapi_log("ENTERING\n")
static FILE *__cjctapi_logfile = NULL;

#ifdef DEBUG_CJCTAPI
#define cjctapi_log(format, args...) \
	__cjctapi_log(__FILE__, __LINE__, __FUNCTION__, format, ## args)
void __cjctapi_log(char *file, int line, const char *function, 
		   const char *format, ...)
{
	char *timestr;
	va_list ap;
	time_t tm;
	FILE *outfd;

	if (__cjctapi_logfile)
		outfd = __cjctapi_logfile;
	else
		outfd = stderr;
	
	va_start(ap, format);
	tm = time(NULL);
	timestr = ctime(&tm);
	timestr[strlen(timestr)-1] = '\0';
	fprintf(outfd, "%s %s:%s():%d: ", timestr, file, function, line);
	vfprintf(outfd, format, ap);
	va_end(ap);
	fflush(outfd);
}
#else
#define cjctapi_log(format, args...)
#endif

static int usb_read = 0;

static struct usb_device *find_rsct_usbdev(int num)
{
	struct usb_bus *busses, *bus;
	struct usb_device *dev;
	int found = 0;

	cjctapi_enter();

	if (!usb_read) {
		usb_init();
		usb_find_busses();
		usb_find_devices();
		usb_read = 1;
	}

	busses = usb_get_busses();

	/* FIXME: this ignores topology changes after first device was opened */
	for (bus = busses; bus; bus = bus->next) {
		for (dev = bus->devices; dev; dev = dev->next) {
			if (dev->descriptor.idVendor == CJPPA_USB_VENDOR_ID &&
			    (dev->descriptor.idProduct == CJPPA_USB_DEVICE_ID
			     || dev->descriptor.idProduct == CJECOM_USB_DEVICE_ID)) {
				found++;
				if (found == num) 
					return dev;
			}
		}
	}
	return NULL;
}


IS8 CT_init(IU16 ctn, IU16 pn)
{
	int ret = CT_API_RV_OK;
	int type;
	void *dev;
	struct usb_device *udev;
	char filename[PATH_MAX+1];

	cjctapi_enter();

	if (ctn_list_lookup(ctn, &dev) > 0) {
		cjctapi_log("ctn already exists, cannot reuse\n");
		return CT_API_RV_ERR_INVALID;
	}

	udev = find_rsct_usbdev(pn);
	if (!udev) {
		cjctapi_log("no matching usb device found\n");
		return CT_API_RV_ERR_INVALID;
	}

	switch (udev->descriptor.idProduct) {
		case CJPPA_USB_DEVICE_ID:
			cjctapi_log("detected pinpad_a at %s\n", 
				    udev->filename);
			type = CJ_TYPE_PPA;
			dev = ctapiInit(udev->filename, NULL, NULL, NULL);
			cjctapi_log("ctapiInit returned %p\n", dev);
			if (!dev)
				ret = CT_API_RV_ERR_HOST;
			break;
		case CJECOM_USB_DEVICE_ID:
			snprintf(filename, PATH_MAX, "/dev/ttyUSB%u", pn-1);
			cjctapi_log("detected e-com/pp at %s, assuming %s\n", 
				    udev->filename, filename);
			type = CJ_TYPE_ECOMPP;
			ret = cjecom_CT_init(filename,
					     (struct cj_info **)&dev);
			cjctapi_log("cjecom_CT_init returned %d\n", ret);
			break;
		default:
			cjctapi_log("unknown Device ID 0x%x found\n",
				    udev->descriptor.idProduct);
			ret = CT_API_RV_ERR_INVALID;
			break;
	}

	if (ret == 0) {
		if (ctn_list_add(ctn, dev, type) < 0) {
			if (type == CJ_TYPE_PPA) 
				ctapiClose(dev);
			else if (type == CJ_TYPE_ECOMPP)
				cjecom_CT_close(dev);
			cjctapi_log("unable to add ctn %u to list\n", ctn);
			return CT_API_RV_ERR_HOST;
		}
	}

	return ret;
}

IS8 CT_data(IU16 ctn, IU8 *dad, IU8 *sad, IU16 lenc, IU8 *command, 
	    IU16 *lenr, IU8 *response)
{
	int ret;
	void *dev;
	int type = ctn_list_lookup(ctn, &dev);

	if (type < 0)
		return CT_API_RV_ERR_INVALID;

	switch (type) {
	case CJ_TYPE_ECOMPP:
		ret = cjecom_CT_data(dev, dad, sad, lenc, command, lenr,
				     response);
		break;
	case CJ_TYPE_PPA:
		ret = ctapiData(dev, dad, sad, lenc, command, lenr, response);
		break;
	}

	return ret;
}

IS8 CT_close(IU16 ctn)
{
	int ret;
	void *dev;
	int type = ctn_list_lookup(ctn, &dev);

	IU8 dad=CT_API_AD_CT, sad=CT_API_AD_HOST, response[2];
	IU16 lenr=sizeof(response);

	if (type < 0)
		return CT_API_RV_ERR_INVALID;

	/* EJECT ICC */
	CT_data(ctn, &dad, &sad, 4, "\x20\x15\x01\x07", &lenr, response);

	switch(type) {
	case CJ_TYPE_ECOMPP:
		ret = cjecom_CT_close(dev);
		break;
	case CJ_TYPE_PPA:
		ret = ctapiClose(dev);
		break;
	}
	ctn_list_del(ctn);
	return ret;
}

/* Proprietary extension */
IS8 CT_keycb( IU16 ctn, void (* cb)(void) )
{
	int ret;
	void *dev;
	int type = ctn_list_lookup(ctn, &dev);

	if (type < 0)
		return CT_API_RV_ERR_INVALID;

	switch (type) {
	case CJ_TYPE_ECOMPP:
		ret = cjecom_CT_keycb(dev, cb);
		break;
	default:
		/* not implemented */
		ret = CT_API_RV_ERR_INVALID;
		break;
	}

	return ret;
}
