/***************************************************************************
 * CT-API library for the REINER SCT cyberJack pinpad/e-com USB.
 * Copyright (C) 2004  REINER SCT
 * Author: Harald Welte
 * Support: linux-usb@sii.li
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * File: ctn_list.c
 * CVS: $Id:$
 ***************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
static pthread_mutex_t ctn_list_lock = PTHREAD_MUTEX_INITIALIZER;
#else
#define pthread_mutex_lock(x)
#define pthread_mutex_unlock(x)
#endif

/* internal houskeeping of cjppa device pointers and CTAPI ctn */

struct ctn_list_item {
	struct ctn_list_item *next;
	void *dev;
	u_int16_t ctn;
	u_int16_t type;
	void (*key_cb)(u_int16_t ctn, unsigned char status);
};

static struct ctn_list_item *ctn_list = NULL;

static struct ctn_list_item *__ctn_list_lookup(u_int16_t ctn)
{
	struct ctn_list_item *cur;

	for (cur = ctn_list; cur; cur = cur->next)
		if (cur->ctn == ctn)
			return cur;
	return NULL;
}

int ctn_list_lookup(u_int16_t ctn, void **dev)
{
	struct ctn_list_item *cli;
	int type;

	pthread_mutex_lock(&ctn_list_lock);
	cli = __ctn_list_lookup(ctn);

	if (!cli) {
		pthread_mutex_unlock(&ctn_list_lock);
		return -ENOENT;
	}
	if (dev)
		*dev = cli->dev;
	type = cli->type;
	pthread_mutex_unlock(&ctn_list_lock);

	return type;
}

u_int16_t ctn_list_lookup_bydev(void *dev)
{
	struct ctn_list_item *cli;
	u_int16_t ret = 0;

	pthread_mutex_lock(&ctn_list_lock);
	for (cli = ctn_list; cli; cli = cli->next)
		if (cli->dev == dev)
			ret = cli->ctn;
	pthread_mutex_unlock(&ctn_list_lock);

	return ret;
}

int ctn_list_add(u_int16_t ctn, void *dev, u_int16_t type)
{
	struct ctn_list_item *cli = malloc(sizeof(*cli));
	if (!cli)
		return -ENOMEM;

	if (type == 0)
		return -EINVAL;

	pthread_mutex_lock(&ctn_list_lock);
	if (__ctn_list_lookup(ctn) != NULL) {
		pthread_mutex_unlock(&ctn_list_lock);
		free(cli);
		return -EEXIST;
	}

	memset(cli, 0, sizeof(*cli));
	cli->ctn = ctn;
	cli->dev = dev;
	cli->type = type;

	cli->next = ctn_list;
	ctn_list = cli;
	pthread_mutex_unlock(&ctn_list_lock);
	return 0;
}

int ctn_list_del(u_int16_t ctn) 
{
	struct ctn_list_item *cur, *prev = NULL;

	pthread_mutex_lock(&ctn_list_lock);
	for (cur = ctn_list; cur; cur = cur->next) {
		if (cur->ctn == ctn) {
			if (prev)
				prev->next = cur->next;
			else
				ctn_list = cur->next;
			pthread_mutex_unlock(&ctn_list_lock);
			free(cur);
			return 0;
		}
		prev = cur;
	}
	pthread_mutex_unlock(&ctn_list_lock);
	return -ENODEV;
}

int ctn_list_set_keycb(u_int16_t ctn, 
		       void (*key_cb)(u_int16_t ctn, unsigned char status))
{
	struct ctn_list_item *cli;

	pthread_mutex_lock(&ctn_list_lock);
	cli = __ctn_list_lookup(ctn);
	if (!cli) {
		pthread_mutex_unlock(&ctn_list_lock);
		return -ENODEV;
	}
	cli->key_cb = key_cb;
	pthread_mutex_unlock(&ctn_list_lock);

	return 0;
}

int ctn_list_empty()
{
	if (ctn_list == NULL)
		return 1;

	return 0;
}
