/***************************************************************************
 * CT-API library for the REINER SCT cyberJack pinpad/e-com USB.
 * Copyright (C) 2001  REINER SCT
 * Author: Matthias Bruestle, Harald Welte
 * Support: linux-usb@sii.li
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * File: ctsh.c
 * CVS: $Id:$
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#if defined(HAVE_READLINE_HISTORY_H)
#include <readline/history.h>
#endif
#if defined(HAVE_READLINE_READLINE_H)
#include <readline/readline.h>
#endif
#include <ctapi.h>

#define PROGRAM	"CT-API Shell - Copyright 2002-2004 by REINER SCT\n"
#define SCRIPTFILE	".ctshrc"
#define MAX_COMMAND_SIZE	700
#define VERBOSE

#if defined(__WIN16__) || defined(__WIN32__)
#ifndef WINDOWS
#define WINDOWS
#endif  /* WINDOWS */
#else
#define HAS_DLOPEN
#endif

#if defined(HAS_DLOPEN)
#include <dlfcn.h>
#elif defined(__WIN32__)
#include <windows.h>
#endif /* HAS_DLOPEN */

#ifndef FALSE
#define FALSE   0
#endif
#ifndef TRUE
#define TRUE    !0
#endif

#define printarray( name, length, array ); \
	printf(name); \
	for( i=0; i<length; i++ ) printf(" %.2X",array[i]); \
	printf("\n");

typedef struct ctapi_info {
	int		bcsver;		/* CT-BCS version */
	IU8		atr[32];
	int		atrlen;
} CT_API_RV_INFO;

#if defined(HAS_DLOPEN)
#define CT_API_RV_CALL
#elif defined(__WIN32__)
#define CT_API_RV_CALL __stdcall
#endif /* HAS_DLOPEN */

/* Function prototypes */
signed char CT_API_RV_CALL CT_init( IU16 ctn, IU16 pn );
signed char CT_API_RV_CALL CT_data( IU16 ctn, IU8 *dad, IU8 *sad, IU16 lenc,
	IU8 *command, IU16 *lenr, IU8 *response );
signed char CT_API_RV_CALL CT_close( IU16 ctn );

/* Global function pointers. */
typedef signed char (CT_API_RV_CALL *CT_INIT)( IU16 ctn, IU16 pn );
typedef signed char (CT_API_RV_CALL *CT_DATA)( IU16 ctn, IU8 *dad, IU8 *sad,
	IU16 lenc, IU8 *command, IU16 *lenr, IU8 *response );
typedef signed char (CT_API_RV_CALL *CT_CLOSE)( IU16 ctn );
static CT_INIT pCT_init = NULL;
static CT_DATA pCT_data = NULL;
static CT_CLOSE pCT_close = NULL;
#define CT_init pCT_init
#define CT_data pCT_data
#define CT_close pCT_close

#if defined(HAS_DLOPEN)
	static void *LibHandle = NULL;
#elif defined(WINDOWS)
#define NULL_HINSTANCE	(HINSTANCE) NULL
	static HINSTANCE LibHandle = NULL_HINSTANCE;
#endif /* HAS_DLOPEN */

void unloadlib( void )
{
#if defined(HAS_DLOPEN)
	dlclose( LibHandle );
#elif defined(WINDOWS)
	FreeLibrary( LibHandle );
#endif /* HAS_DLOPEN */
}

int loadlib( char *libname )
{
#ifdef HAS_DLOPEN
	LibHandle = dlopen( libname, RTLD_LAZY );
	if( !LibHandle ) {
		printf(" Error.\n");
		return(-1);
	}

	pCT_init = (CT_INIT) dlsym( LibHandle, "CT_init" );
	if( dlerror()!=NULL ) {
		unloadlib( );
		printf(" Error.\n");
		return(-1);
	}
	pCT_data = (CT_DATA) dlsym( LibHandle, "CT_data" );
	if( dlerror()!=NULL ) {
		unloadlib( );
		printf(" Error.\n");
		return(-1);
	}
	pCT_close = (CT_CLOSE) dlsym( LibHandle, "CT_close" );
	if( dlerror()!=NULL ) {
		unloadlib( );
		printf(" Error.\n");
		return(-1);
	}
#elif defined(WINDOWS)
	if( ( LibHandle = LoadLibrary( argv[1] ) ) == NULL_HINSTANCE ) {
		printf(" Error.\n");
		return(-1);
	}

	pCT_init = (CT_INIT) GetProcAddress( LibHandle, "CT_init" );
	pCT_data = (CT_DATA) GetProcAddress( LibHandle, "CT_data" );
	pCT_close = (CT_CLOSE) GetProcAddress( LibHandle, "CT_close" );

	/* Make sure we got valid pointers for every card function */
	if( pCT_init == NULL || pCT_data == NULL || pCT_close == NULL ) {
		unloadlib( );
		printf(" Error.\n");
		return(-1);
	}
#endif /* HAS_DLOPEN */

	return 0;
}

/* Eject ICC */

int eject( IU16 ctn )
{
	IU8 cmd[]={0x20,0x15,0x01,0x00};
	IU8 rsp[260];
	IU8 sad=2, dad=1;
	IU16 lenc=4, lenr=16;
	IS8 ret;

	ret = (*CT_data)( ctn, &dad, &sad, lenc, cmd, &lenr, rsp );

	if( (ret!=CT_API_RV_OK) || (lenr!=2) ) {
		return(FALSE);
	}

	return(TRUE);
}

void strcompact( char *str )
{
	int i,j=0;

	for( i=0; i<strlen(str); i++ ) {
		if( !isspace(str[i]) ) str[j++]=tolower( str[i] );
	}

	str[j]=0;
}

int cmdcheck( char *str )
{
	int i, start;

	if( strlen(str)<4 ) return FALSE;

	if( (str[2]<0x30) || (str[2]>0x39) ) return FALSE;

	if( str[3]==':' ) {
		if( strlen(str)&1 ) return FALSE;
		start=4;
	} else if( str[4]==':' ) {
		if( (str[3]<0x30) || (str[3]>0x39) ) return FALSE;
		if( !(strlen(str)&1) ) return FALSE;
		start=5;
	} else {
		return FALSE;
	}
	
	for( i=start; i<strlen(str); i++ ) {
		if( !isxdigit(str[i]) ) return FALSE;
	}

	return TRUE;
}

int strtobin( char *s, char *d, IU16 *len )
{
	long l;
	int i, ret;

	if( *s==':' ) s++;

	for(i=0; i<(strlen(s)>>1); i++ ) {
		if( i>=(CJ_CTAPI_MAX_LENC) ) return FALSE;
		ret=sscanf( s+(i<<1), "%2lx", &l );
		if( ret!=1 ) return FALSE;
		d[i]=l&0xFF;
	}

	*len=i;

	return TRUE;
}

int sendCmd( IU16 ctn, char *command )
{
	IU8 cmd[CJ_CTAPI_MAX_LENC];
	IU8 rsp[CJ_CTAPI_MAX_LENR];
	IU8 sad=2, dad;
	IU16 lenc, lenr=sizeof(rsp);
	IS8 ret;
	int i;

	if( !cmdcheck( command ) ) {
		printf( "Command format error.\n" );
		return FALSE;
	}

	if( !strtobin( command+4, cmd, &lenc ) ) {
		printf( "Command decoding error.\n" );
		return FALSE;
	}

	if( command[3]==':' ) {
		dad = command[2]&0x0F;
	} else {
		dad = (command[2]&0x0F)*10+(command[3]&0x0F);
	}

	printf( "Command to %d", dad );
	printarray( ":", lenc, cmd );

	ret = (*CT_data)( ctn, &dad, &sad, lenc, cmd, &lenr, rsp );

	printf( "Ret: %d\n", ret );
	if( ret==CT_API_RV_OK ) {
		printf( "Response from %d", sad );
		printarray( ":", lenr, rsp );
	}

	if( (ret!=CT_API_RV_OK) || (lenr!=2) ) {
		return FALSE;
	}

	return TRUE;
}

int list( void )
{
	FILE *scrfile;
	char *home=getenv("HOME");
	char *name;
	char line[MAX_COMMAND_SIZE];

	/* Allocate memory for total filename */
	name = malloc( strlen(home)+1+strlen(SCRIPTFILE)+1 );
	if( name==NULL ) {
		printf("Malloc error.\n");
		return FALSE;
	}

	strcpy( name, home );
	strcat( name, "/" );
	strcat( name, SCRIPTFILE );

	scrfile=fopen( name, "r" );
	if( scrfile==NULL ) {
		free( name );
		printf("Unable to open command file.\n");
		return FALSE;
	}

	/* Search for command */
	while( fgets( line, sizeof(line), scrfile )!=NULL ) {
		printf( line );
	}

	fclose( scrfile );
	return TRUE;
}

int script( IU16 ctn, char *command )
{
	FILE *scrfile;
	char *home=getenv("HOME");
	char *name;
	char line[MAX_COMMAND_SIZE];
	char *data=NULL;
	char *extradata=NULL;
	char *sendcmd;
	int length;
	int ret;

	/* Skip "co" */
	command+=2;

	/* Allocate memory for total filename */
	name = malloc( strlen(home)+1+strlen(SCRIPTFILE)+1 );
	if( name==NULL ) {
		printf("Malloc error.\n");
		return FALSE;
	}

	strcpy( name, home );
	strcat( name, "/" );
	strcat( name, SCRIPTFILE );

	scrfile=fopen( name, "r" );
	if( scrfile==NULL ) {
		free( name );
		printf("Unable to open command file.\n");
		return FALSE;
	}

	/* Look for extra data in command data */
	extradata = strchr( command, ':' );
	if( extradata!=NULL ) *extradata++=0;

	/* Search for command */
	while( fgets( line, sizeof(line), scrfile )!=NULL ) {
		/* Look for EOF */
		data = strchr( line, '\n' );
		if( data==NULL ) {
			printf("Error in script file. Probably line to long.\n");
			fclose( scrfile );
			return FALSE;
		}
		*data=0; /* Cut off */

		/* Look for begin of data */
		data = strchr( line, ' ' );
		if( data!=NULL ) *data++=0;

		/* Correct command? */
		if( strcmp( line, command )!=0 ) continue;

		/* Not needed anymore. */
		fclose( scrfile );

		/* Get send command length */
		length=2+strlen(data)+1;
		if(extradata!=NULL) length+=strlen(extradata);
		sendcmd=malloc( length );
		if( sendcmd==NULL ) {
			printf("Malloc error.\n");
			fclose( scrfile );
			return FALSE;
		}

		/* Assemble send command */
		strcpy( sendcmd, "se" );
		strcat( sendcmd, data );
		if( extradata!=NULL ) strcat( sendcmd, extradata );
		strcompact( sendcmd );

		/* Send command */
		ret = sendCmd( ctn, sendcmd );
		free( sendcmd );
		return ret;
	}

	return FALSE;
}

void help( )
{
	printf( PROGRAM );
	printf( "ej - eject card\n" );
	printf( "co cmd[:xxxx...] - send command from script (xx: hex data appended to command)\n" );
	printf( "se d:xxxxxxxx... - send APDU (d: dad, xx: data as hex, spaces are ignored)\n" );
	printf( "qu - quit\n" );
	printf( "he - help\n" );
}

int main( int argc, char *argv[] )
{
	IU16 ctn=1, pn;
	IS8 ret;
	int tmp;
#if defined(HAVE_READLINE_HISTORY_H) && defined(HAVE_READLINE_READLINE_H)
	char *command=NULL;
	HIST_ENTRY **the_history_list;
#else
	char command[MAX_COMMAND_SIZE];
#endif

	printf( PROGRAM );

	if( argc!=3 ) {
		printf("Wrong number of arguments.\n");
		return(-1);
	}

	if( (sscanf( argv[2], "%d", &tmp )!=1) || (tmp<0) ) {
		printf("Bad argument 2.\n");
		return(-1);
	}

	pn=tmp & 0xFFFF;

	printf("Loading CT-API library (%s):",argv[1]);
	if( loadlib( argv[1] )!= 0 ) return -1;
	printf(" Ok.\n");

	/* Init */
	printf("Initialising CT-API library with pn %d:", pn );
	ret = (*CT_init)( ctn, pn );
	if( ret!=CT_API_RV_OK ) {
		unloadlib( );
		printf(" Error. (%d)\n",ret);
		return(-1);
	}
	printf(" Ok.\n");

#if defined(HAVE_READLINE_HISTORY_H) && defined(HAVE_READLINE_READLINE_H)
	using_history();
	stifle_history( 100 );
#endif

	/* Command line loop */
	while( 1 ) {
#if defined(HAVE_READLINE_HISTORY_H) && defined(HAVE_READLINE_READLINE_H)
		if( command!=NULL ) free(command);
		command = readline( "ctsh> " );
		if( command==NULL ) continue;
		add_history( command );
#else
		printf( "ctsh> " );
		fgets( command, MAX_COMMAND_SIZE-1, stdin );
#endif
		
		strcompact( command );

		/* Help */
		if( strncmp( "ej", command, 2 )==0 ) {
			eject( ctn );
			continue;
		}

		/* Command from script */
		if( strncmp( "co", command, 2 )==0 ) {
			script( ctn, command );
			continue;
		}

		/* Send APDU */
		if( strncmp( "se", command, 2 )==0 ) {
			sendCmd( ctn, command );
			continue;
		}

		/* List commands */
		if( strncmp( "li", command, 2 )==0 ) {
			list( );
			continue;
		}

		/* Help */
		if( strncmp( "he", command, 2 )==0 ) {
			help( );
			continue;
		}

		/* Quit */
		if( strncmp( "qu", command, 2 )==0 ) {
			printf( "Mahlzeit.\n" );
			break;
		}

		/* Exit */
		if( strncmp( "ex", command, 2 )==0 ) {
			printf( "Mahlzeit.\n" );
			break;
		}

		printf( "Command not recognized. Type 'he' for help.\n" );
	}

	/* Close */
	printf("Closing CT-API library:");
	ret = (*CT_close)( ctn );
	if( ret!=CT_API_RV_OK ) {
		unloadlib( );
		printf(" Error. (%d)\n",ret);
		return(-1);
	}
	printf(" Ok.\n");

	return(0);
}

